/**
 * Inline example tab panel. Allows code to be demonstrated and edited inline.
 */
Ext.define('Docs.view.examples.Inline', {
    extend: 'Ext.Panel',
    alias: 'widget.inlineexample',
    requires: [
        'Docs.view.examples.InlineEditor',
        'Docs.view.examples.InlinePreview'
    ],

    componentCls: 'inline-example-cmp',
    layout: 'card',
    border: 0,
    resizable: {
        transparent: true,
        handles: 's',
        constrainTo: false
    },
    // Make too long examples scrollable
    maxCodeHeight: 400,

    /**
     * @cfg {Object} options
     * A set of options for configuring the preview:
     *
     * @cfg {String} options.device phone, miniphone or tablet
     * @cfg {String} options.orientation ladscape or portrait
     * @cfg {Boolean} options.raw True to turn off Ext.setup().
     * @cfg {Boolean} options.preview True to start up in preview mode.
     */
    options: {},

    /**
     * @cfg {Docs.view.examples.InlineToolbar} toolbar
     * The toolbar with buttons that controls this component.
     */

    initComponent: function() {
        this.options = Ext.apply({
            device: "phone",
            orientation: "landscape"
        }, this.options);

        this.items = [
            this.editor = Ext.create('Docs.view.examples.InlineEditor', {
                cmpName: 'code',
                value: this.value,
                listeners: {
                    change: this.updateHeight,
                    scope: this
                }
            }),
            this.preview = Ext.create('Docs.view.examples.InlinePreview', {
                cmpName: 'preview',
                options: this.options
            })
        ];

        if (this.options.preview) {
            this.activeItem = 1;
            this.toolbar.activateButton("preview");
        }
        else {
            this.activeItem = 0;
            this.toolbar.activateButton("code");
        }

        this.on("afterrender", this.init, this);

        this.callParent(arguments);
    },

    // Updates code inside example component
    init: function() {
        var activeItem = this.layout.getActiveItem();
        if (activeItem.cmpName === 'preview') {
            this.showPreview();
        }
        this.updateHeight();
        this.initToolbarEvents();
    },

    initToolbarEvents: function() {
        this.toolbar.on("buttonclick", function(name) {
            if (name === "code") {
                this.showCode();
            }
            else if (name === "preview") {
                this.showPreview();
            }
            else if (name === "copy") {
                this.showCode();
                this.editor.selectAll();
            }
        }, this);
    },

    /**
     * Activates the code card.
     */
    showCode: function() {
        this.layout.setActiveItem(0);
        this.updateHeight();
        this.toolbar.activateButton("code");
        // Weird bug on CodeMirror requires 2 refreshes...
        this.editor.refresh();
        this.editor.refresh();
    },

    /**
     * Activates the preview card.
     */
    showPreview: function() {
        this.preview.update(this.editor.getValue());
        this.layout.setActiveItem(1);
        this.updateHeight();
        this.toolbar.activateButton("preview");
    },

    // Syncs the height with number of lines in code example.
    updateHeight: function() {
        var previewHeight = this.preview.getHeight();
        var editorHeight = this.editor.getHeight();
        var toolbarHeight = 30;
        if (Docs.data.touchExamplesUi && previewHeight > 0) {
            this.setHeight(previewHeight+toolbarHeight);
        }
        else if (editorHeight > 0) {
            this.setHeight(Ext.Number.constrain(editorHeight+toolbarHeight, 0, this.maxCodeHeight));
        }
    }

});
